/*****************************************************************************
*
* File Name  : wm_wdrt_demo.c
*
* Description: wifi direct rx or tx demo
*
* Copyright (c) 2014 Winner Micro Electronic Design Co., Ltd.
* All rights reserved.
*
* Author     : 
*
* Date : 2019-5-6
*****************************************************************************/
#include <string.h>
#include "wm_include.h"
#include "wm_ieee80211.h"
#include "tls_common.h"
#include "tls_wireless.h"
#include "wm_wdrt_demo.h"

extern u8 *wpa_supplicant_get_mac(void);
extern struct tls_wif * tls_get_wif_data(void);

#define WM_WDRT_WAIT_ACK_CNT_MAX       5
#define WM_WDRT_WAIT_ACK_TIMEOUT       15 /* 30ms */

const static u8 wm_wdrt_eth_llc[8] = {0xAA, 0xAA, 0x03, 0x28, 0x6D, 0xCD, 0xEE, 0xEF};

static int wm_wdrt_last_seq = -1;
static u8 wm_wdrt_last_sa_mac[ETH_ALEN];
static wm_wdrt_rx_callback wm_wdrt_demo_rx_callback = NULL;

static u8 wm_wdrt_local_mac[6];

static u32 wm_wdrt_rx_seq = 0;
static u32 wm_wdrt_tx_seq = 1;
static u32 wm_wdrt_tx_ack_seq = 0;
static tls_os_sem_t *wm_wdrt_tx_ack_sem = NULL;

static int wm_wdrt_build_data_ack_frame(const u8 *dst_mac, u32 seq_no)
{
    int ret;
    u8 *packet;
    u32 packet_len;
    struct tls_wifi_tx_rate_t tx;
    struct ieee80211_hdr_3addr *hdr;

    //memset(&tx, 0, sizeof(tx));
    tx.tx_rate = WM_WIFI_TX_RATEIDX_1M;
    tx.tx_gain = tls_wifi_get_tx_gain_max(tx.tx_rate);

    /* hdr + llc + seq-no */
    packet_len = sizeof(struct ieee80211_hdr_3addr) + sizeof(wm_wdrt_eth_llc) + 4;
    packet = tls_mem_alloc(packet_len);
    if (!packet)
        return -10;

    //memset(packet, 0, packet_len);
    hdr = (struct ieee80211_hdr_3addr *)packet;
    hdr->frame_control = host_to_le16(IEEE80211_FTYPE_DATA | IEEE80211_STYPE_DATA);

    memcpy(hdr->addr1, dst_mac, 6);
    memcpy(hdr->addr2, wm_wdrt_local_mac, 6);
    memcpy(hdr->addr3, wm_wdrt_local_mac, 6);

    memcpy(packet + sizeof(struct ieee80211_hdr_3addr), wm_wdrt_eth_llc, sizeof(wm_wdrt_eth_llc));
    memcpy(packet + sizeof(struct ieee80211_hdr_3addr) + sizeof(wm_wdrt_eth_llc), &seq_no, 4);

    ret = tls_wifi_send_data(NULL, packet, packet_len, &tx);

    tls_mem_free(packet);

    return ret;
}

static void wm_wdrt_demo_recv_callback(u8* data, u32 data_len)
{
    u32 rx_seq = 0;
    struct ieee80211_hdr_3addr *hdr;

    if (data_len < (sizeof(struct ieee80211_hdr_3addr) + sizeof(wm_wdrt_eth_llc) + 4))
        return;

    hdr = (struct ieee80211_hdr_3addr *)data;

    if (ieee80211_has_fromds(hdr->frame_control) ||
        ieee80211_has_tods(hdr->frame_control))
        return;

    if (!ieee80211_is_data(hdr->frame_control))
        return;

    if (ieee80211_is_data_qos(hdr->frame_control))
        return;

    /* local_mac, src_mac, src_mac */
    if (compare_ether_addr(hdr->addr1, wm_wdrt_local_mac) &&
        compare_ether_addr(hdr->addr2, hdr->addr3))
        return;

    if (ieee80211_has_retry(hdr->frame_control))
    {
        if (-1 != wm_wdrt_last_seq)
        {
            if ((hdr->seq_ctrl == wm_wdrt_last_seq) &&
                (0 == compare_ether_addr(hdr->addr2, wm_wdrt_last_sa_mac)))
                return;
        }
    }

    wm_wdrt_last_seq = hdr->seq_ctrl;
    memcpy(wm_wdrt_last_sa_mac, hdr->addr2, ETH_ALEN);

    data += sizeof(struct ieee80211_hdr_3addr);
    data_len -= sizeof(struct ieee80211_hdr_3addr);

    if (memcmp(data, wm_wdrt_eth_llc, sizeof(wm_wdrt_eth_llc)))
        return;

    data += sizeof(wm_wdrt_eth_llc);
    data_len -= sizeof(wm_wdrt_eth_llc);

    if (4 == data_len)
    {
        memcpy(&wm_wdrt_tx_ack_seq, data, 4);
        tls_os_sem_release(wm_wdrt_tx_ack_sem);
    }
    else
    {
        memcpy(&rx_seq, data, 4);

        if (wm_wdrt_rx_seq != rx_seq) /* filter retry */
        {
            wm_wdrt_build_data_ack_frame(hdr->addr2, rx_seq);

            if (wm_wdrt_demo_rx_callback)
                wm_wdrt_demo_rx_callback(hdr->addr2, data + 4, data_len - 4);

            wm_wdrt_rx_seq = rx_seq;
        }
    }

    return;
}

static int wm_wdrt_send_data(u8 *dst_mac, u8 *data, u16 data_len, enum tls_wifi_tx_rate tx_rate)
{
    int ret;
    u8 *packet;
    u32 packet_len;
    u8 ack_wait_cnt = 0;
    struct tls_wifi_tx_rate_t tx;
    struct ieee80211_hdr_3addr *hdr;

    //memset(&tx, 0, sizeof(tx));
    tx.tx_rate = tx_rate;
    tx.tx_gain = tls_wifi_get_tx_gain_max(tx.tx_rate);

    packet_len = sizeof(struct ieee80211_hdr_3addr) + sizeof(wm_wdrt_eth_llc) + 4 + data_len;
    packet = tls_mem_alloc(packet_len);
    if (!packet)
        return -10;

    //memset(packet, 0, packet_len);
    hdr = (struct ieee80211_hdr_3addr *)packet;
    hdr->frame_control = host_to_le16(IEEE80211_FTYPE_DATA | IEEE80211_STYPE_DATA);
    memcpy(hdr->addr1, dst_mac, 6);
    memcpy(hdr->addr2, wm_wdrt_local_mac, 6);
    memcpy(hdr->addr3, wm_wdrt_local_mac, 6);

    memcpy(packet + sizeof(struct ieee80211_hdr_3addr), wm_wdrt_eth_llc, sizeof(wm_wdrt_eth_llc));
    memcpy(packet + sizeof(struct ieee80211_hdr_3addr) + sizeof(wm_wdrt_eth_llc), &wm_wdrt_tx_seq, 4);
    memcpy(packet + sizeof(struct ieee80211_hdr_3addr) + sizeof(wm_wdrt_eth_llc) + 4, data, data_len);

    while (ack_wait_cnt++ < WM_WDRT_WAIT_ACK_CNT_MAX)
    {
        ret = tls_wifi_send_data(NULL, packet, packet_len, &tx);
        if (ret)
            break;

        /* wait ack */    
        ret = tls_os_sem_acquire(wm_wdrt_tx_ack_sem, WM_WDRT_WAIT_ACK_TIMEOUT);
        if (!ret)
        {
            if (wm_wdrt_tx_ack_seq == wm_wdrt_tx_seq)
            {
                ret = 0;
                break;
            }
            else
            {
                ret = -11;
                if (ack_wait_cnt == 2) /* 0-1 *M, 2-4 1M */
                {
                    tx.tx_rate = WM_WIFI_TX_RATEIDX_1M;
                    tx.tx_gain = tls_wifi_get_tx_gain_max(tx.tx_rate);
                }

                printf("retry %hhu send seq %u rate = %d.\r\n", 
                        ack_wait_cnt, wm_wdrt_tx_seq, tx.tx_rate);
            }
        }
    }

    tls_mem_free(packet);
    wm_wdrt_tx_seq++;

    return ret;
}

static void wm_wdrt_demo_entry_tx_loop(u8 *dst_mac, u8 *data, u16 data_len, enum tls_wifi_tx_rate tx_rate)
{
    int ret;

    while (1)
    {
        ret = wm_wdrt_send_data(dst_mac, data, data_len, tx_rate);
        if (ret)
            printf("failed to send data\r\n");
    }
}

void wm_wdrt_demo(wm_wdrt_cfg *cfg)
{
	printf("\n wifi direct %s demo\n", (cfg->mode == WM_WDRT_MODE_RX) ? "rx" : "tx");

	tls_get_mac_addr(wm_wdrt_local_mac);

    wm_wdrt_last_seq = -1;

    wm_wdrt_rx_seq = 0;
	wm_wdrt_tx_seq = 1;
    wm_wdrt_tx_ack_seq = 0;

    wm_wdrt_demo_rx_callback = NULL;

    if (!wm_wdrt_tx_ack_sem)
        tls_os_sem_create(&wm_wdrt_tx_ack_sem, 0);

    if (0  != cfg->work_channel)
        tls_wifi_change_chanel(cfg->work_channel - 1);
    else
        tls_wifi_change_chanel(13);

    tls_wl_if_set_mode(tls_get_wif_data(), IEEE80211_MODE_IBSS);
    tls_wifi_data_recv_cb_register((tls_wifi_data_recv_callback)wm_wdrt_demo_recv_callback);

    if (WM_WDRT_MODE_TX == cfg->mode)
    {
        wm_wdrt_demo_entry_tx_loop(cfg->da, cfg->tx_data, cfg->tx_data_len, cfg->tx_rate);
    }
    else if (WM_WDRT_MODE_RX == cfg->mode)
    {
        wm_wdrt_demo_rx_callback = cfg->rx_callback;
    }

    return;
}

